<?php
session_start();
require_once '../config/db.php';
require_once '../auth/auth.php';
require_once '../helpers/helpers.php';

// Set active page for navigation
$activePage = 'suppliers';

// Check if user is logged in
requireLogin();

// Check if user has permission to view suppliers
// Check if user has any supplier-related permissions
$supplierPermissions = [
    'view_suppliers', 'manage_suppliers', 'add_supplier', 'edit_supplier', 'delete_supplier',
    'search_suppliers', 'process_supplier_payment', 'add_supplier_item'
];

if (!hasAnyPermission($supplierPermissions)) {
    $_SESSION['error'] = "You don't have permission to view supplier details.";
    header('Location: dashboard.php');
    exit;
}

if (false && !hasPermission('view_suppliers')) { // Disable the old check
    $_SESSION['error'] = "You don't have permission to view supplier details.";
    header('Location: suppliers.php');
    exit;
}

// Get supplier ID from URL
$supplier_id = isset($_GET['id']) ? (int)$_GET['id'] : 0;

if (!$supplier_id) {
    $_SESSION['error'] = "Invalid supplier ID.";
    header('Location: suppliers.php');
    exit;
}

// Get supplier details
$stmt = $pdo->prepare("
    SELECT s.*, 
           COUNT(DISTINCT sr.id) as total_receipts,
           SUM(sr.quantity) as total_items,
           SUM(sr.quantity * sr.unit_price) as total_value
    FROM suppliers s
    LEFT JOIN supplier_receipts sr ON s.id = sr.supplier_id
    WHERE s.id = ?
    GROUP BY s.id
");
$stmt->execute([$supplier_id]);
$supplier = $stmt->fetch();

if (!$supplier) {
    $_SESSION['error'] = "Supplier not found.";
    header('Location: suppliers.php');
    exit;
}

// Get recent receipts
$stmt = $pdo->prepare("
    SELECT sr.*, p.name as product_name
    FROM supplier_receipts sr
    JOIN products p ON sr.product_id = p.id
    WHERE sr.supplier_id = ?
    LIMIT 10
");
$stmt->execute([$supplier_id]);
$recent_receipts = $stmt->fetchAll();

// Get currency from store settings
$stmt = $pdo->prepare("SELECT setting_value FROM store_settings WHERE setting_key = 'currency'");
$stmt->execute();
$currency = $stmt->fetchColumn() ?: 'Ksh';
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Supplier Details - EDDUH COMPUTERS</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <style>
        body {
            background: #f8fafc;
        }
        .card {
            border: none;
            border-radius: 15px;
            box-shadow: 0 4px 6px -1px rgba(0, 0, 0, 0.1), 0 2px 4px -1px rgba(0, 0, 0, 0.06);
        }
        .nav-tabs {
            border: none;
            margin-bottom: 1.5rem;
        }
        .nav-tabs .nav-link {
            border: none;
            color: #6b7280;
            padding: 0.75rem 1.5rem;
            border-radius: 8px;
            margin-right: 0.5rem;
            font-weight: 500;
        }
        .nav-tabs .nav-link:hover {
            background-color: #f3f4f6;
        }
        .nav-tabs .nav-link.active {
            background-color: #3b82f6;
            color: white;
        }
        .info-card {
            background: white;
            border-radius: 12px;
            padding: 1.5rem;
            margin-bottom: 1rem;
        }
        .info-label {
            color: #6b7280;
            font-size: 0.875rem;
            margin-bottom: 0.25rem;
        }
        .info-value {
            color: #111827;
            font-size: 1rem;
            font-weight: 500;
        }
        .stat-card {
            background: white;
            border-radius: 12px;
            padding: 1.5rem;
            text-align: center;
        }
        .stat-value {
            font-size: 1.5rem;
            font-weight: 600;
            color: #111827;
            margin-bottom: 0.5rem;
        }
        .stat-label {
            color: #6b7280;
            font-size: 0.875rem;
        }
        .btn-action {
            padding: 0.5rem 1rem;
            border-radius: 8px;
            font-weight: 500;
        }
        .table {
            margin-bottom: 0;
        }
        .table th {
            font-weight: 600;
            color: #374151;
            border-bottom-width: 1px;
        }
        .table td {
            vertical-align: middle;
        }
        
        /* Modal Styles */
        .modal-content {
            border: none;
            border-radius: 15px;
            box-shadow: 0 10px 15px -3px rgba(0, 0, 0, 0.1), 0 4px 6px -2px rgba(0, 0, 0, 0.05);
        }
        .modal-header {
            border-bottom: 1px solid #e5e7eb;
            padding: 1.25rem 1.5rem;
            background: #f8fafc;
            border-radius: 15px 15px 0 0;
        }
        .modal-body {
            padding: 1.5rem;
        }
        .modal-footer {
            border-top: 1px solid #e5e7eb;
            padding: 1.25rem 1.5rem;
            background: #f8fafc;
            border-radius: 0 0 15px 15px;
        }
        .form-label {
            font-weight: 500;
            color: #374151;
            margin-bottom: 0.5rem;
        }
        .form-control {
            border-radius: 8px;
            border: 1px solid #e5e7eb;
            padding: 0.6rem 1rem;
            transition: all 0.2s ease;
        }
        .form-control:focus {
            border-color: #3b82f6;
            box-shadow: 0 0 0 3px rgba(59, 130, 246, 0.1);
        }
        .required-field::after {
            content: "*";
            color: #ef4444;
            margin-left: 4px;
        }
        .btn-modal {
            padding: 0.6rem 1.5rem;
            border-radius: 8px;
            font-weight: 500;
            transition: all 0.2s ease;
        }
        .btn-modal-primary {
            background-color: #3b82f6;
            border: none;
            color: white;
        }
        .btn-modal-primary:hover {
            background-color: #2563eb;
            transform: translateY(-1px);
        }
        .btn-modal-secondary {
            background-color: #6b7280;
            border: none;
            color: white;
        }
        .btn-modal-secondary:hover {
            background-color: #4b5563;
            transform: translateY(-1px);
        }
        .invalid-feedback {
            font-size: 0.875rem;
            color: #ef4444;
            margin-top: 0.25rem;
        }
        .alert {
            border: none;
            border-radius: 8px;
            padding: 1rem 1.25rem;
            margin-bottom: 1rem;
        }
        .alert-success {
            background-color: #ecfdf5;
            color: #065f46;
        }
        .alert-danger {
            background-color: #fef2f2;
            color: #991b1b;
        }
    </style>
</head>
<body>
    <div class="container-fluid py-4">
        <div class="row">
            <?php include '../includes/sidebar.php'; ?>
            <div class="col-md-10 ms-sm-auto px-md-4">
                <?php include '../includes/topbar.php'; ?>

                <!-- Header -->
                <div class="d-flex justify-content-between align-items-center mb-4">
                    <div>
                        <h2 class="mb-1">Supplier Details</h2>
                        <p class="text-muted mb-0">View and manage supplier information</p>
                    </div>
                    <div>
                        <?php if (hasPermission('manage_suppliers')): ?>
                        <button type="button" class="btn btn-primary btn-action me-2" data-bs-toggle="modal" data-bs-target="#editSupplierModal">
                            <i class="fas fa-edit me-2"></i>Edit Supplier
                        </button>
                        <?php endif; ?>
                        <a href="suppliers.php" class="btn btn-secondary btn-action">
                            <i class="fas fa-arrow-left me-2"></i>Back to Suppliers
                        </a>
                    </div>
                </div>

                <!-- Stats Cards -->
                <div class="row mb-4">
                    <div class="col-md-4">
                        <div class="stat-card">
                            <div class="stat-value"><?php echo number_format($supplier['total_receipts']); ?></div>
                            <div class="stat-label">Total Receipts</div>
                        </div>
                    </div>
                    <div class="col-md-4">
                        <div class="stat-card">
                            <div class="stat-value"><?php echo number_format($supplier['total_items'] ?? 0); ?></div>
                            <div class="stat-label">Total Items</div>
                        </div>
                    </div>
                    <div class="col-md-4">
                        <div class="stat-card">
                            <div class="stat-value"><?php echo $currency . ' ' . number_format($supplier['total_value'] ?? 0, 2); ?></div>
                            <div class="stat-label">Total Value</div>
                        </div>
                    </div>
                </div>

                <!-- Tabs -->
                <ul class="nav nav-tabs" id="supplierTabs" role="tablist">
                    <li class="nav-item" role="presentation">
                        <button class="nav-link active" id="info-tab" data-bs-toggle="tab" data-bs-target="#info" type="button" role="tab">
                            <i class="fas fa-info-circle me-2"></i>Information
                        </button>
                    </li>
                    <li class="nav-item" role="presentation">
                        <button class="nav-link" id="receipts-tab" data-bs-toggle="tab" data-bs-target="#receipts" type="button" role="tab">
                            <i class="fas fa-receipt me-2"></i>Recent Receipts
                        </button>
                    </li>
                </ul>

                <!-- Tab Content -->
                <div class="tab-content" id="supplierTabsContent">
                    <!-- Information Tab -->
                    <div class="tab-pane fade show active" id="info" role="tabpanel">
                        <div class="row">
                            <div class="col-md-6">
                                <div class="info-card">
                                    <div class="info-label">Supplier Name</div>
                                    <div class="info-value"><?php echo htmlspecialchars($supplier['name']); ?></div>
                                </div>
                            </div>
                            <div class="col-md-6">
                                <div class="info-card">
                                    <div class="info-label">Contact Person</div>
                                    <div class="info-value"><?php echo htmlspecialchars($supplier['contact_person']); ?></div>
                                </div>
                            </div>
                            <div class="col-md-6">
                                <div class="info-card">
                                    <div class="info-label">Phone</div>
                                    <div class="info-value"><?php echo htmlspecialchars($supplier['phone']); ?></div>
                                </div>
                            </div>
                            <div class="col-md-6">
                                <div class="info-card">
                                    <div class="info-label">Email</div>
                                    <div class="info-value"><?php echo htmlspecialchars($supplier['email']); ?></div>
                                </div>
                            </div>
                            <div class="col-md-6">
                                <div class="info-card">
                                    <div class="info-label">Address</div>
                                    <div class="info-value"><?php echo htmlspecialchars($supplier['address']); ?></div>
                                </div>
                            </div>
                            <div class="col-md-6">
                                <div class="info-card">
                                    <div class="info-label">Tax Number</div>
                                    <div class="info-value"><?php echo htmlspecialchars($supplier['tax_number']); ?></div>
                                </div>
                            </div>
                        </div>
                    </div>

                    <!-- Receipts Tab -->
                    <div class="tab-pane fade" id="receipts" role="tabpanel">
                        <div class="card">
                            <div class="card-body">
                                <div class="table-responsive">
                                    <table class="table">
                                        <thead>
                                            <tr>
                                                <th>Date</th>
                                                <th>Product</th>
                                                <th>Quantity</th>
                                                <th>Unit Price</th>
                                                <th>Total</th>
                                            </tr>
                                        </thead>
                                        <tbody>
                                            <?php foreach ($recent_receipts as $receipt): ?>
                                            <tr>
                                                <td><?php echo date('Y-m-d', strtotime($receipt['created_at'])); ?></td>
                                                <td><?php echo htmlspecialchars($receipt['product_name']); ?></td>
                                                <td><?php echo number_format($receipt['quantity']); ?></td>
                                                <td><?php echo $currency . ' ' . number_format($receipt['unit_price'], 2); ?></td>
                                                <td><?php echo $currency . ' ' . number_format($receipt['quantity'] * $receipt['unit_price'], 2); ?></td>
                                            </tr>
                                            <?php endforeach; ?>
                                            <?php if (empty($recent_receipts)): ?>
                                            <tr>
                                                <td colspan="5" class="text-center">No receipts found</td>
                                            </tr>
                                            <?php endif; ?>
                                        </tbody>
                                    </table>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- Edit Supplier Modal -->
    <div class="modal fade" id="editSupplierModal" tabindex="-1" aria-labelledby="editSupplierModalLabel" aria-hidden="true">
        <div class="modal-dialog modal-lg">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title" id="editSupplierModalLabel">Edit Supplier</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                </div>
                <form id="editSupplierForm" method="post" class="needs-validation" novalidate>
                    <div class="modal-body">
                        <div id="modalAlert"></div>
                        <div class="row g-3">
                            <div class="col-md-6">
                                <label for="name" class="form-label required-field">Supplier Name</label>
                                <input type="text" class="form-control" id="name" name="name" 
                                       value="<?php echo htmlspecialchars($supplier['name']); ?>" required>
                                <div class="invalid-feedback">Please enter supplier name.</div>
                            </div>

                            <div class="col-md-6">
                                <label for="contact_person" class="form-label">Contact Person</label>
                                <input type="text" class="form-control" id="contact_person" name="contact_person"
                                       value="<?php echo htmlspecialchars($supplier['contact_person']); ?>">
                            </div>

                            <div class="col-md-6">
                                <label for="phone" class="form-label">Phone</label>
                                <input type="tel" class="form-control" id="phone" name="phone"
                                       value="<?php echo htmlspecialchars($supplier['phone']); ?>">
                            </div>

                            <div class="col-md-6">
                                <label for="email" class="form-label">Email</label>
                                <input type="email" class="form-control" id="email" name="email"
                                       value="<?php echo htmlspecialchars($supplier['email']); ?>">
                            </div>

                            <div class="col-md-6">
                                <label for="address" class="form-label">Address</label>
                                <textarea class="form-control" id="address" name="address" rows="3"><?php echo htmlspecialchars($supplier['address']); ?></textarea>
                            </div>

                            <div class="col-md-6">
                                <label for="tax_number" class="form-label">Tax Number</label>
                                <input type="text" class="form-control" id="tax_number" name="tax_number"
                                       value="<?php echo htmlspecialchars($supplier['tax_number']); ?>">
                            </div>
                        </div>
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-modal btn-modal-secondary" data-bs-dismiss="modal">Cancel</button>
                        <button type="submit" class="btn btn-modal btn-modal-primary">
                            <i class="fas fa-save me-2"></i>Save Changes
                        </button>
                    </div>
                </form>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        // Form validation
        (function () {
            'use strict'
            var forms = document.querySelectorAll('.needs-validation')
            Array.prototype.slice.call(forms).forEach(function (form) {
                form.addEventListener('submit', function (event) {
                    if (!form.checkValidity()) {
                        event.preventDefault()
                        event.stopPropagation()
                    }
                    form.classList.add('was-validated')
                }, false)
            })
        })()

        // Handle form submission
        document.getElementById('editSupplierForm').addEventListener('submit', function(e) {
            e.preventDefault();
            if (this.checkValidity()) {
                const formData = new FormData(this);
                const submitButton = this.querySelector('button[type="submit"]');
                const originalButtonText = submitButton.innerHTML;
                
                // Disable submit button and show loading state
                submitButton.disabled = true;
                submitButton.innerHTML = '<i class="fas fa-spinner fa-spin me-2"></i>Saving...';

                fetch('edit_supplier.php?id=<?php echo $supplier_id; ?>', {
                    method: 'POST',
                    body: formData
                })
                .then(response => response.json())
                .then(data => {
                    // Reset button state
                    submitButton.disabled = false;
                    submitButton.innerHTML = originalButtonText;

                    // Show alert
                    const alertDiv = document.getElementById('modalAlert');
                    alertDiv.innerHTML = `
                        <div class="alert alert-${data.success ? 'success' : 'danger'} alert-dismissible fade show">
                            ${data.message}
                            <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
                        </div>
                    `;

                    if (data.success) {
                        // Close modal and reload page after 1 second
                        setTimeout(() => {
                            bootstrap.Modal.getInstance(document.getElementById('editSupplierModal')).hide();
                            window.location.reload();
                        }, 1000);
                    }
                })
                .catch(error => {
                    // Reset button state
                    submitButton.disabled = false;
                    submitButton.innerHTML = originalButtonText;

                    // Show error alert
                    const alertDiv = document.getElementById('modalAlert');
                    alertDiv.innerHTML = `
                        <div class="alert alert-danger alert-dismissible fade show">
                            An error occurred while saving the changes. Please try again.
                            <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
                        </div>
                    `;
                    console.error('Error:', error);
                });
            }
        });

        // Clear alerts when modal is closed
        document.getElementById('editSupplierModal').addEventListener('hidden.bs.modal', function () {
            document.getElementById('modalAlert').innerHTML = '';
            document.getElementById('editSupplierForm').classList.remove('was-validated');
        });
    </script>
    <?php include '../includes/footer.php'; ?>
</body>
</html> 